<?php
/**
 * Chat gateway: receives conversation history from WordPress,
 * forwards it to OpenAI, and returns the assistant reply.
 */
declare(strict_types=1);

define('WERADMIN_GATEWAY', true);

header('Content-Type: application/json; charset=utf-8');

require_once __DIR__ . '/config.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    respond([
        'error'   => 'invalid_method',
        'message' => 'Only POST is allowed.',
    ], 405);
}

$token = trim($_SERVER['HTTP_X_WER_KEY'] ?? '');
if ($token === '' || ! isset($WERADMIN_GATEWAY_CONFIG['allowed_tokens'][$token])) {
    respond([
        'error'   => 'unauthorized',
        'message' => 'Missing or invalid license token.',
    ], 401);
}

$rawBody = file_get_contents('php://input');
$payload = json_decode($rawBody, true);

if (! is_array($payload)) {
    respond([
        'error'   => 'invalid_json',
        'message' => 'Request body must be valid JSON.',
    ], 400);
}

$conversationId = isset($payload['conversation_id']) ? (string) $payload['conversation_id'] : '';
$messages       = isset($payload['messages']) && is_array($payload['messages']) ? $payload['messages'] : [];

if (empty($messages)) {
    respond([
        'error'   => 'missing_messages',
        'message' => 'No conversation messages were provided.',
    ], 400);
}

// Build OpenAI chat messages.
$systemPrompt = $WERADMIN_GATEWAY_CONFIG['system_prompt'] ?? 'You are a warm, proactive customer support specialist. Greet visitors politely, provide clear answers, and always ask if you can help with anything else.';

$openaiMessages   = [
    [
        'role'    => 'system',
        'content' => $systemPrompt,
    ],
];
$allowedSenders = [
    'visitor'   => 'user',
    'user'      => 'user',
    'assistant' => 'assistant',
    'agent'     => 'assistant',
    'admin'     => 'assistant',
    'human'     => 'assistant',
    'system'    => 'system',
];

foreach ($messages as $entry) {
    $text = isset($entry['message']) ? (string) $entry['message'] : '';
    $text = strip_tags(html_entity_decode($text, ENT_QUOTES | ENT_HTML5));
    if ($text === '') {
        continue;
    }

    if (isset($entry['role'])) {
        $role = (string) $entry['role'];
    } else {
        $sender = strtolower((string) ($entry['sender'] ?? 'visitor'));
        $role   = $allowedSenders[$sender] ?? 'user';
    }

    $openaiMessages[] = [
        'role'    => $role,
        'content' => $text,
    ];
}

$requestPayload = [
    'model'       => $WERADMIN_GATEWAY_CONFIG['openai_model'] ?? 'gpt-4o-mini',
    'temperature' => 0.3,
    'max_tokens'  => 320,
    'messages'    => $openaiMessages,
];

// Call OpenAI.
$response = call_openai(
    $WERADMIN_GATEWAY_CONFIG['openai_api_key'],
    $requestPayload
);

if ($response['status'] >= 400) {
    respond([
        'error'   => 'openai_error',
        'message' => $response['body']['error']['message'] ?? 'OpenAI request failed.',
        'status'  => $response['status'],
    ], 502);
}

$reply = $response['body']['choices'][0]['message']['content'] ?? '';
$reply = trim($reply);

if ($reply === '') {
    respond([
        'error'   => 'empty_reply',
        'message' => 'OpenAI returned an empty response.',
    ], 502);
}

respond([
    'conversation_id' => $conversationId,
    'reply'           => $reply,
    'usage'           => $response['body']['usage'] ?? null,
]);

/**
 * Send a JSON response and exit.
 */
function respond(array $data, int $status = 200): void
{
    http_response_code($status);
    echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

/**
 * Call the OpenAI Chat Completions API via cURL.
 *
 * @return array{status:int,body:array|string}
 */
function call_openai(string $apiKey, array $payload): array
{
    $ch = curl_init('https://api.openai.com/v1/chat/completions');

    curl_setopt_array($ch, [
        CURLOPT_POST           => true,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER     => [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $apiKey,
        ],
        CURLOPT_POSTFIELDS     => json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES),
        CURLOPT_TIMEOUT        => 20,
    ]);

    $body      = curl_exec($ch);
    $curlError = curl_error($ch);
    $status    = curl_getinfo($ch, CURLINFO_HTTP_CODE);

    curl_close($ch);

    if ($body === false) {
        return [
            'status' => 500,
            'body'   => ['error' => ['message' => 'Curl error: ' . $curlError]],
        ];
    }

    $decoded = json_decode($body, true);

    if (json_last_error() !== JSON_ERROR_NONE) {
        $decoded = ['raw' => $body];
    }

    return [
        'status' => $status,
        'body'   => $decoded,
    ];
}
